// SPDX-License-Identifier: GPL-2.0+
/*
 * Marvell 10G x3510 PHY driver
 *
 */
#include <linux/bitfield.h>
#include <linux/ctype.h>
#include <linux/delay.h>
#include <linux/hwmon.h>
#include <linux/marvell_phy.h>
#include <linux/phy.h>
#include <linux/sfp.h>
#include <linux/netdevice.h>

enum {
	MV_PMA_FW_VER0		= 0xc011,
	MV_PMA_FW_VER1		= 0xc012,

	MV_PMA_BOOT		= 0xc050,
	MV_PMA_BOOT_FATAL	= BIT(0),

	MV_PCS_BASE_T		= 0x0000,

	MV_PCS_CSCR1		= 0x8000,
	MV_PCS_CSCR1_ED_MASK	= 0x0300,
	MV_PCS_CSCR1_ED_OFF	= 0x0000,
	MV_PCS_CSCR1_ED_RX	= 0x0200,
	MV_PCS_CSCR1_ED_NLP	= 0x0300,
	MV_PCS_CSCR1_MDIX_MASK	= 0x0060,
	MV_PCS_CSCR1_MDIX_MDI	= 0x0000,
	MV_PCS_CSCR1_MDIX_MDIX	= 0x0020,
	MV_PCS_CSCR1_MDIX_AUTO	= 0x0060,

	MV_PCS_CSSR1		= 0x8008,
	MV_PCS_CSSR1_SPD1_MASK	= 0xc000,
	MV_PCS_CSSR1_SPD1_SPD2	= 0xc000,
	MV_PCS_CSSR1_SPD1_1000	= 0x8000,
	MV_PCS_CSSR1_SPD1_100	= 0x4000,
	MV_PCS_CSSR1_SPD1_10	= 0x0000,
	MV_PCS_CSSR1_DUPLEX_FULL= BIT(13),
	MV_PCS_CSSR1_RESOLVED	= BIT(11),
	MV_PCS_CSSR1_MDIX	= BIT(6),
	MV_PCS_CSSR1_SPD2_MASK	= 0x000c,
	MV_PCS_CSSR1_SPD2_5000	= 0x0008,
	MV_PCS_CSSR1_SPD2_2500	= 0x0004,
	MV_PCS_CSSR1_SPD2_10000	= 0x0000,

	/* Number of ports on the device */
	MV_PCS_PORT_INFO	= 0xd00d,
	MV_PCS_PORT_INFO_NPORTS_MASK	= 0x0380,
	MV_PCS_PORT_INFO_NPORTS_SHIFT	= 7,

	/* These registers appear at 0x800X and 0xa00X - the 0xa00X control
	 * registers appear to set themselves to the 0x800X when AN is
	 * restarted, but status registers appear readable from either.
	 */
	MV_AN_CTRL1000		= 0x8000, /* 1000base-T control register */
	MV_AN_STAT1000		= 0x8001, /* 1000base-T status register */

	/* Vendor2 MMD registers */
	MV_V2_PORT_CTRL		= 0xf001,
	MV_V2_PORT_CTRL_PWRDOWN					= BIT(11),
	MV_V2_33X0_PORT_CTRL_SWRST				= BIT(15),
};

static int mv3510_power_down(struct phy_device *phydev)
{
	return phy_set_bits_mmd(phydev, MDIO_MMD_VEND2, MV_V2_PORT_CTRL,
				MV_V2_PORT_CTRL_PWRDOWN);
}

static int mv3510_power_up(struct phy_device *phydev)
{
	int ret;

	ret = phy_clear_bits_mmd(phydev, MDIO_MMD_VEND2, MV_V2_PORT_CTRL,
				 MV_V2_PORT_CTRL_PWRDOWN);

	/* Sometimes, the power down bit doesn't clear immediately, and
	 * a read of this register causes the bit not to clear. Delay
	 * 100us to allow the PHY to come out of power down mode before
	 * the next access.
	 */
	udelay(100);

	return phy_set_bits_mmd(phydev, MDIO_MMD_VEND2, MV_V2_PORT_CTRL,
				MV_V2_33X0_PORT_CTRL_SWRST);
}

static int mv3510_reset(struct phy_device *phydev, u32 unit)
{
	int val, err;

	err = phy_modify_mmd(phydev, MDIO_MMD_PCS, unit + MDIO_CTRL1,
			     MDIO_CTRL1_RESET, MDIO_CTRL1_RESET);
	if (err < 0)
		return err;

	return phy_read_mmd_poll_timeout(phydev, MDIO_MMD_PCS,
					 unit + MDIO_CTRL1, val,
					 !(val & MDIO_CTRL1_RESET),
					 5000, 100000, true);
}

static int mv3510_probe(struct phy_device *phydev)
{
	u32 mmd_mask = MDIO_DEVS_PMAPMD | MDIO_DEVS_AN;
	int ret;
	u32 firmware_ver;

	if (!phydev->is_c45 ||
	    (phydev->c45_ids.devices_in_package & mmd_mask) != mmd_mask)
		return -ENODEV;

	ret = phy_read_mmd(phydev, MDIO_MMD_PMAPMD, MV_PMA_BOOT);
	if (ret < 0)
		return ret;

	if (ret & MV_PMA_BOOT_FATAL) {
		dev_warn(&phydev->mdio.dev,
			 "PHY failed to boot firmware, status=%04x\n", ret);
		return -ENODEV;
	}

	ret = phy_read_mmd(phydev, MDIO_MMD_PMAPMD, MV_PMA_FW_VER0);
	if (ret < 0)
		return ret;

	firmware_ver = ret << 16;

	ret = phy_read_mmd(phydev, MDIO_MMD_PMAPMD, MV_PMA_FW_VER1);
	if (ret < 0)
		return ret;

	firmware_ver |= ret;

	phydev_info(phydev, "Firmware version %u.%u.%u.%u\n",
		    firmware_ver >> 24, (firmware_ver >> 16) & 255,
		    (firmware_ver >> 8) & 255, firmware_ver & 255);

	/* Powering down the port when not in use saves about 600mW */
	ret = mv3510_power_down(phydev);
	if (ret)
		return ret;

	return ret;
}

static int mv3510_suspend(struct phy_device *phydev)
{
	return mv3510_power_down(phydev);
}

static int mv3510_resume(struct phy_device *phydev)
{
	int ret;

	ret = mv3510_power_up(phydev);
	if (ret)
		return ret;

	return ret;
}

static int mv3510_config_init(struct phy_device *phydev)
{
	int err;

	phydev->mdix_ctrl = ETH_TP_MDI_AUTO;

	/* Power up so reset works */
	err = mv3510_power_up(phydev);
	if (err)
		return err;

	return 0;
}

static int mv3510_config_mdix(struct phy_device *phydev)
{
	u16 val;
	int err;

	switch (phydev->mdix_ctrl) {
	case ETH_TP_MDI_AUTO:
		val = MV_PCS_CSCR1_MDIX_AUTO;
		break;
	case ETH_TP_MDI_X:
		val = MV_PCS_CSCR1_MDIX_MDIX;
		break;
	case ETH_TP_MDI:
		val = MV_PCS_CSCR1_MDIX_MDI;
		break;
	default:
		return -EINVAL;
	}

	err = phy_modify_mmd_changed(phydev, MDIO_MMD_PCS, MV_PCS_CSCR1,
				     MV_PCS_CSCR1_MDIX_MASK, val);
	if (err > 0)
		err = mv3510_reset(phydev, MV_PCS_BASE_T);

	return err;
}

static int mv3510_config_aneg(struct phy_device *phydev)
{
	bool changed = false;
	u16 reg;
	int ret;

	ret = mv3510_config_mdix(phydev);
	if (ret < 0)
		return ret;

	if (phydev->autoneg == AUTONEG_DISABLE)
		return genphy_c45_pma_setup_forced(phydev);

	ret = genphy_c45_an_config_aneg(phydev);
	if (ret < 0)
		return ret;
	if (ret > 0)
		changed = true;

	/* Clause 45 has no standardized support for 1000BaseT, therefore
	 * use vendor registers for this mode.
	 */
	reg = linkmode_adv_to_mii_ctrl1000_t(phydev->advertising);
	ret = phy_modify_mmd_changed(phydev, MDIO_MMD_AN, MV_AN_CTRL1000,
			     ADVERTISE_1000FULL | ADVERTISE_1000HALF, reg);
	if (ret < 0)
		return ret;
	if (ret > 0)
		changed = true;

	return genphy_c45_check_and_restart_aneg(phydev, changed);
}

static int mv3510_aneg_done(struct phy_device *phydev)
{
	return genphy_c45_aneg_done(phydev);
}

static void mv3510_clear_status(struct phy_device *phydev)
{
	phydev->speed = SPEED_UNKNOWN;
	phydev->duplex = DUPLEX_UNKNOWN;
	linkmode_zero(phydev->lp_advertising);
	phydev->link = 0;
	phydev->pause = 0;
	phydev->asym_pause = 0;
	phydev->mdix = ETH_TP_MDI_INVALID;
}

static int mv3510_read_status_copper(struct phy_device *phydev)
{
	int cssr1, speed, val;

	val = genphy_c45_read_link(phydev);
	if (val < 0)
		return val;

	val = phy_read_mmd(phydev, MDIO_MMD_AN, MDIO_STAT1);
	if (val < 0)
		return val;

	cssr1 = phy_read_mmd(phydev, MDIO_MMD_PCS, MV_PCS_CSSR1);
	if (cssr1 < 0)
		return cssr1;

	/* If the link settings are not resolved, mark the link down */
	if (!(cssr1 & MV_PCS_CSSR1_RESOLVED)) {
		mv3510_clear_status(phydev);
		return 0;
	}

	/* Read the copper link settings */
	speed = cssr1 & MV_PCS_CSSR1_SPD1_MASK;
	if (speed == MV_PCS_CSSR1_SPD1_SPD2)
		speed |= cssr1 & MV_PCS_CSSR1_SPD2_MASK;

	switch (speed) {
	case MV_PCS_CSSR1_SPD1_SPD2 | MV_PCS_CSSR1_SPD2_10000:
		phydev->speed = SPEED_10000;
		break;

	case MV_PCS_CSSR1_SPD1_SPD2 | MV_PCS_CSSR1_SPD2_5000:
		phydev->speed = SPEED_5000;
		break;

	case MV_PCS_CSSR1_SPD1_SPD2 | MV_PCS_CSSR1_SPD2_2500:
		phydev->speed = SPEED_2500;
		break;

	case MV_PCS_CSSR1_SPD1_1000:
		phydev->speed = SPEED_1000;
		break;

	case MV_PCS_CSSR1_SPD1_100:
		phydev->speed = SPEED_100;
		break;

	case MV_PCS_CSSR1_SPD1_10:
		phydev->speed = SPEED_10;
		break;
	}

	phydev->duplex = cssr1 & MV_PCS_CSSR1_DUPLEX_FULL ?
			 DUPLEX_FULL : DUPLEX_HALF;
	phydev->port = PORT_TP;
	phydev->mdix = cssr1 & MV_PCS_CSSR1_MDIX ?
		       ETH_TP_MDI_X : ETH_TP_MDI;

	if (val & MDIO_AN_STAT1_COMPLETE) {
		val = genphy_c45_read_lpa(phydev);
		if (val < 0)
			return val;

		/* Read the link partner's 1G advertisement */
		val = phy_read_mmd(phydev, MDIO_MMD_AN, MV_AN_STAT1000);
		if (val < 0)
			return val;

		mii_stat1000_mod_linkmode_lpa_t(phydev->lp_advertising, val);

		/* Update the pause status */
		phy_resolve_aneg_pause(phydev);
	}

	return 0;
}

static int mv3510_read_status(struct phy_device *phydev)
{
	int err;

	err = mv3510_read_status_copper(phydev);
	if (err < 0) {
		mv3510_clear_status(phydev);
		return err;
	}

	return 0;
}

static int mv3510_get_number_of_ports(struct phy_device *phydev)
{
	int ret;

	ret = phy_read_mmd(phydev, MDIO_MMD_PCS, MV_PCS_PORT_INFO);
	if (ret < 0)
		return ret;

	ret &= MV_PCS_PORT_INFO_NPORTS_MASK;
	ret >>= MV_PCS_PORT_INFO_NPORTS_SHIFT;

	return ret + 1;
}

static int mv3510_match_phy_device(struct phy_device *phydev)
{
	if ((phydev->c45_ids.device_ids[MDIO_MMD_PMAPMD] &
	     MARVELL_PHY_ID_MASK) != MARVELL_PHY_ID_88X3510)
		return 0;

	return mv3510_get_number_of_ports(phydev) == 1;
}

static struct phy_driver mv3510_drivers[] = {
	{
		.phy_id		= MARVELL_PHY_ID_88X3510,
		.phy_id_mask	= MARVELL_PHY_ID_MASK,
		.match_phy_device = mv3510_match_phy_device,
		.name		= "mv88x3510",
		.features	= PHY_10GBIT_FULL_FEATURES,
		.config_init	= mv3510_config_init,
		.probe		= mv3510_probe,
		.suspend	= mv3510_suspend,
		.resume		= mv3510_resume,
		.config_aneg	= mv3510_config_aneg,
		.aneg_done	= mv3510_aneg_done,
		.read_status	= mv3510_read_status,
		.set_loopback	= genphy_c45_loopback,
	},
};

module_phy_driver(mv3510_drivers);

static struct mdio_device_id __maybe_unused mv3510_tbl[] = {
	{ MARVELL_PHY_ID_88X3510, MARVELL_PHY_ID_MASK },
	{ },
};
MODULE_DEVICE_TABLE(mdio, mv3510_tbl);
MODULE_DESCRIPTION("Marvell x3510 Ethernet PHY driver");
MODULE_LICENSE("GPL");
