// SPDX-License-Identifier: GPL-2.0+
/*
 * Qualcomm Generic Interface (GENI) Serial Engine (SE) Wrapper
 *
 * Copyright (C) 2023 Linaro Ltd. <vladimir.zapolskiy@linaro.org>
 */

#include <common.h>
#include <dm.h>
#include <misc.h>
#include <asm/io.h>
#ifdef CONFIG_QCOM_GENI_SE_FW_LOAD
#include <geni_se.h>

#define SE_GENI_INIT_CFG_REVISION_OFFSET	0x0000
#define SE_GENI_S_INIT_CFG_REVISION_OFFSET	0x0004
#define SE_GENI_FORCE_DEFAULT_REG_OFFSET	0x0020
#define SE_GENI_OUTPUT_CTRL_OFFSET		0x0024
#define SE_GENI_CGC_CTRL_OFFSET			0x0028
#define SE_GENI4_IMAGE_REG_OFFSET		0x0100
#define SE_HW_PARAM_2_OFFSET			0x0E2C
#define SE_GENI_FW_REVISION_OFFSET		0x1000
#define SE_GENI_S_FW_REVISION_OFFSET		0x1004
#define SE_GENI_CFG_RAMn_OFFSET			0x1010
#define SE_GENI_CLK_CTRL_OFFSET			0x2000
#define SE_DMA_IF_EN_OFFSET			0x2004
#define SE_GENI_USE_MINICORES_MASK		BIT(12)

typedef struct __attribute__ ((packed)) se_fw_hdr
{
    uint32_t firmware_magic;
    /* Structure version number */
    uint32_t se_fw_hdr_version;
    /* QUPV3_HW_VERSION */
    uint32_t qup_core_version;
    /* Programmed into GENI_FW_REVISION */
    uint16_t serial_protocol;
    /* Programmed into GENI_FW_REVISION */
    uint16_t firmware_version;
    /* Programmed into GENI_INIT_CFG_REVISION */
    uint16_t config_version;
    /* Number of (uint32) GENI_FW_RAM words */
    uint16_t firmware_sz_in_items;
    /* Byte offset of GENI_FW_RAM array */
    uint16_t firmware_offset;
    /* Number of GENI_FW_CFG index/value pairs */
    uint16_t config_sz_in_items;
    /* Byte offset of GENI_FW_CFG indexes array */
    uint16_t config_idx_offset;
    /* Byte offset of GENI_FW_CFG values array */
    uint16_t config_val_offset;
} se_fw_hdr;

static uint8_t uart_fw[] __attribute__ ((aligned (4))) = {
    0x53, 0x45, 0x46, 0x57, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x30, 0x02, 0x00, 0x05, 0x04,
    0x0a, 0x00, 0xc4, 0x01, 0x1c, 0x00, 0x45, 0x00, 0x48, 0x08, 0x34, 0x07, 0x32, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x0f, 0x00,
    0x20, 0x00, 0x78, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x42, 0x00, 0xc8, 0xc6, 0x7f, 0x00, 0x00, 0x00, 0x3f, 0x00,
    0xc8, 0xc6, 0x7f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x21, 0x38, 0x7f, 0x00, 0xa1, 0x0b, 0x7f, 0x00, 0x82, 0x38, 0x7f, 0x00,
    0x49, 0xd0, 0x9f, 0x00, 0x0d, 0x2e, 0x7f, 0x00, 0x63, 0x78, 0x9f, 0x00, 0x4f, 0xd0, 0x9f, 0x00,
    0x15, 0x2e, 0x7f, 0x00, 0x63, 0x78, 0x9f, 0x00, 0x57, 0xd0, 0x9f, 0x00, 0x45, 0x10, 0x99, 0x00,
    0x1d, 0x2e, 0x7f, 0x00, 0x63, 0x78, 0x9f, 0x00, 0x5f, 0x50, 0x9f, 0x00, 0x4b, 0x10, 0x99, 0x00,
    0x25, 0x2e, 0x7f, 0x00, 0x63, 0x78, 0x9f, 0x00, 0x53, 0x10, 0x99, 0x00, 0x2d, 0x2e, 0x7f, 0x00,
    0x6d, 0x10, 0x99, 0x00, 0x7f, 0x48, 0x9f, 0x00, 0xe7, 0x00, 0x9f, 0x00, 0x67, 0x18, 0x9f, 0x00,
    0x7f, 0x78, 0x9f, 0x00, 0xe7, 0x00, 0x9f, 0x00, 0x6d, 0x60, 0x9f, 0x00, 0x03, 0x3a, 0x7f, 0x00,
    0x79, 0x48, 0x9f, 0x00, 0x6c, 0x18, 0x9f, 0x00, 0x7f, 0x78, 0x9f, 0x00, 0xe7, 0x00, 0x9f, 0x00,
    0x7f, 0x18, 0x9f, 0x00, 0x79, 0x68, 0x9f, 0x00, 0xa1, 0x0b, 0x7f, 0x00, 0xe7, 0x00, 0x9f, 0x00,
    0x83, 0x60, 0x9f, 0x00, 0x28, 0x34, 0x7f, 0x00, 0x00, 0x08, 0x1f, 0x00, 0x04, 0x34, 0x7f, 0x00,
    0x07, 0x39, 0x7f, 0x00, 0x20, 0x38, 0x7f, 0x00, 0x8f, 0x60, 0x9f, 0x00, 0x00, 0x80, 0x04, 0x00,
    0x07, 0x39, 0x7f, 0x00, 0x8f, 0x50, 0x9f, 0x00, 0xed, 0x08, 0x9f, 0x00, 0x99, 0x60, 0x9f, 0x00,
    0x08, 0x34, 0x7f, 0x00, 0xf5, 0x10, 0x99, 0x00, 0x08, 0x39, 0x7f, 0x00, 0x80, 0x1c, 0x7f, 0x00,
    0xb7, 0x70, 0x9f, 0x00, 0xad, 0x00, 0x9a, 0x00, 0x02, 0x1c, 0x7f, 0x00, 0xd9, 0x58, 0x9f, 0x00,
    0xc5, 0x78, 0x9f, 0x00, 0xd9, 0x10, 0x9f, 0x00, 0x07, 0x20, 0x7f, 0x00, 0xe7, 0x00, 0x9f, 0x00,
    0xb1, 0x98, 0x98, 0x00, 0xd9, 0x78, 0x9f, 0x00, 0xd8, 0x30, 0x9f, 0x00, 0xc4, 0x10, 0x9f, 0x00,
    0xd9, 0x78, 0x9f, 0x00, 0xe7, 0x00, 0x9f, 0x00, 0xd1, 0x18, 0x9f, 0x00, 0xbd, 0x68, 0x9f, 0x00,
    0xd1, 0x78, 0x9f, 0x00, 0xc5, 0x60, 0x9f, 0x00, 0xd0, 0x10, 0x99, 0x00, 0x03, 0x3a, 0x7f, 0x00,
    0xe7, 0x00, 0x9f, 0x00, 0xbd, 0x48, 0x9f, 0x00, 0xca, 0x18, 0x9f, 0x00, 0x20, 0x1c, 0x7f, 0x00,
    0x28, 0x34, 0x7f, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x40, 0x1f, 0x00, 0xe7, 0x00, 0x9f, 0x00,
    0xe0, 0x60, 0x9f, 0x00, 0xd8, 0x10, 0x9f, 0x00, 0x63, 0x78, 0x9f, 0x00, 0xdc, 0x18, 0x9f, 0x00,
    0x28, 0x34, 0x7f, 0x00, 0xdd, 0x78, 0x9f, 0x00, 0x01, 0x60, 0x7f, 0x00, 0xe9, 0x60, 0x9f, 0x00,
    0xd3, 0x78, 0x9f, 0x00, 0xed, 0x60, 0x9f, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x39, 0x7f, 0x00,
    0x99, 0x78, 0x9f, 0x00, 0x07, 0x39, 0x7f, 0x00, 0xf7, 0x60, 0x9f, 0x00, 0x18, 0x34, 0x7f, 0x00,
    0x9f, 0x78, 0x9f, 0x00, 0x00, 0x00, 0x0f, 0x00, 0xe1, 0x0b, 0x7f, 0x00, 0x05, 0x01, 0x9f, 0x00,
    0x00, 0x40, 0x1f, 0x00, 0x01, 0x60, 0x7f, 0x00, 0x03, 0x79, 0x9f, 0x00, 0x00, 0x00, 0x0f, 0x00,
    0xa0, 0x04, 0x7f, 0x00, 0x0d, 0xd1, 0x9f, 0x00, 0x00, 0x40, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x0c, 0x00,
    0x10, 0x00, 0x5e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x92, 0x00,
    0x08, 0x80, 0x94, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x7a, 0x7f, 0x00, 0x08, 0x4c, 0x7f, 0x00, 0x01, 0x40, 0x7f, 0x00, 0xf7, 0x79, 0x9d, 0x00,
    0xe9, 0x79, 0x9d, 0x00, 0x41, 0x88, 0x98, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x01, 0x50, 0x7f, 0x00,
    0x05, 0x20, 0x7f, 0x00, 0x03, 0x20, 0x7f, 0x00, 0x04, 0x19, 0x9f, 0x00, 0x31, 0x8a, 0x98, 0x00,
    0xd3, 0x49, 0x9f, 0x00, 0x01, 0x44, 0x6a, 0x00, 0x00, 0x80, 0x0b, 0x00, 0x66, 0xc0, 0x98, 0x00,
    0x58, 0x40, 0x98, 0x00, 0x43, 0x38, 0x7f, 0x00, 0x03, 0x20, 0x7f, 0x00, 0x67, 0x30, 0x9f, 0x00,
    0x62, 0x40, 0x99, 0x00, 0x08, 0x1c, 0x7f, 0x00, 0x10, 0x1c, 0x7f, 0x00, 0x67, 0x78, 0x9f, 0x00,
    0x00, 0x1e, 0x7f, 0x00, 0x04, 0x20, 0x7f, 0x00, 0x6a, 0x38, 0x99, 0x00, 0x04, 0x1c, 0x7f, 0x00,
    0x6b, 0x68, 0x9f, 0x00, 0x07, 0x39, 0x68, 0x00, 0x20, 0x38, 0x7f, 0x00, 0x82, 0xb8, 0x7f, 0x00,
    0x91, 0x50, 0x9f, 0x00, 0x1d, 0x41, 0x9f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x57, 0x11, 0x9f, 0x00,
    0x7b, 0x60, 0x9f, 0x00, 0x01, 0x50, 0x7f, 0x00, 0x9d, 0x48, 0x8b, 0x00, 0x07, 0x39, 0x7f, 0x00,
    0x87, 0x18, 0x9f, 0x00, 0x08, 0x60, 0x7f, 0x00, 0x08, 0x20, 0x7f, 0x00, 0x02, 0x24, 0x7f, 0x00,
    0x8e, 0x29, 0x9a, 0x00, 0x48, 0x4c, 0x7f, 0x00, 0x8f, 0x79, 0x1f, 0x01, 0x95, 0x88, 0x98, 0x00,
    0xdf, 0x79, 0x9d, 0x00, 0x95, 0x60, 0x9f, 0x00, 0x07, 0xb9, 0x68, 0x00, 0x09, 0x20, 0x7f, 0x00,
    0x73, 0x78, 0x9f, 0x00, 0x01, 0x8a, 0x98, 0x00, 0xaa, 0x30, 0x99, 0x00, 0x88, 0x4c, 0x7f, 0x00,
    0x02, 0x60, 0x7f, 0x00, 0xbc, 0x10, 0x9f, 0x00, 0x02, 0x60, 0x7f, 0x00, 0x34, 0x4a, 0x9f, 0x00,
    0xbd, 0x90, 0x98, 0x00, 0xb1, 0x38, 0x9f, 0x00, 0x80, 0x1c, 0x7f, 0x00, 0xbc, 0x00, 0x9a, 0x00,
    0xbc, 0x28, 0x9f, 0x00, 0xb9, 0x18, 0x9f, 0x00, 0x08, 0x60, 0x7f, 0x00, 0x01, 0x20, 0x7f, 0x00,
    0x10, 0x24, 0x7f, 0x00, 0x4c, 0x48, 0x9f, 0x00, 0xc2, 0x38, 0x99, 0x00, 0x04, 0x1c, 0x7f, 0x00,
    0x01, 0x40, 0x7f, 0x00, 0xcb, 0x58, 0x9f, 0x00, 0x4c, 0x48, 0x9f, 0x00, 0x47, 0x78, 0x9f, 0x00,
    0x0a, 0x04, 0x7f, 0x00, 0x4c, 0xc8, 0x9f, 0x00, 0xd2, 0xb8, 0x99, 0x00, 0x04, 0x9c, 0x7f, 0x00,
    0xf4, 0x98, 0x9f, 0x00, 0xd9, 0x88, 0x98, 0x00, 0x13, 0xf9, 0x9f, 0x00, 0x4c, 0xc8, 0x9f, 0x00,
    0xd9, 0xd0, 0x9f, 0x00, 0x4c, 0x48, 0x9f, 0x00, 0x02, 0x1c, 0x7f, 0x00, 0xcb, 0x58, 0x9f, 0x00,
    0xe8, 0x18, 0x9f, 0x00, 0x4c, 0x48, 0x9f, 0x00, 0xcb, 0x78, 0x9f, 0x00, 0x01, 0x1c, 0x7f, 0x00,
    0x07, 0x20, 0x7f, 0x00, 0x10, 0x60, 0x7f, 0x00, 0x02, 0x20, 0x7f, 0x00, 0x4c, 0x48, 0x9f, 0x00,
    0x47, 0x78, 0x9f, 0x00, 0x00, 0xa1, 0x7f, 0x00, 0x00, 0x80, 0x1f, 0x00, 0x00, 0xa2, 0x7f, 0x00,
    0xd4, 0x98, 0x9f, 0x00, 0x10, 0xe0, 0x7f, 0x00, 0x02, 0xa0, 0x7f, 0x00, 0x4c, 0xc8, 0x9f, 0x00,
    0x47, 0xf8, 0x9f, 0x00, 0x00, 0x21, 0x7f, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x22, 0x7f, 0x00,
    0x48, 0x18, 0x9f, 0x00, 0x10, 0x60, 0x7f, 0x00, 0x02, 0x20, 0x7f, 0x00, 0x49, 0x78, 0x9f, 0x00,
    0x18, 0x89, 0x9f, 0x00, 0x4c, 0xc8, 0x8e, 0x00, 0xd9, 0xf8, 0x9f, 0x00, 0x4c, 0xc8, 0x8d, 0x00,
    0xd9, 0xf8, 0x9f, 0x00, 0x39, 0x11, 0x9f, 0x00, 0x1f, 0x61, 0x9f, 0x00, 0x26, 0x21, 0x8b, 0x00,
    0x05, 0x39, 0x7f, 0x00, 0x7b, 0x78, 0x9f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x2c, 0x21, 0x9a, 0x00,
    0x48, 0x4c, 0x7f, 0x00, 0x31, 0x19, 0x9f, 0x00, 0x08, 0x60, 0x7f, 0x00, 0x01, 0x20, 0x7f, 0x00,
    0x08, 0x20, 0x7f, 0x00, 0x01, 0x24, 0x7f, 0x00, 0x7b, 0x78, 0x9f, 0x00, 0x39, 0x61, 0x9f, 0x00,
    0x42, 0x21, 0x8b, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x01, 0x34, 0x7f, 0x00, 0x59, 0x79, 0x9f, 0x00,
    0x05, 0x39, 0x7f, 0x00, 0x49, 0x19, 0x9f, 0x00, 0x08, 0x60, 0x7f, 0x00, 0x01, 0x24, 0x7f, 0x00,
    0x08, 0x20, 0x7f, 0x00, 0x01, 0x20, 0x7f, 0x00, 0x02, 0x34, 0x7f, 0x00, 0x54, 0x21, 0x9a, 0x00,
    0x48, 0x4c, 0x7f, 0x00, 0x59, 0x79, 0x9f, 0x00, 0x01, 0x34, 0x7f, 0x00, 0x59, 0x61, 0x9f, 0x00,
    0x01, 0x50, 0x7f, 0x00, 0x71, 0x49, 0x8b, 0x00, 0x07, 0x39, 0x7f, 0x00, 0x65, 0x19, 0x9f, 0x00,
    0x08, 0x60, 0x7f, 0x00, 0x02, 0x24, 0x7f, 0x00, 0x02, 0x34, 0x7f, 0x00, 0x08, 0x20, 0x7f, 0x00,
    0x8e, 0x29, 0x9a, 0x00, 0x48, 0x4c, 0x7f, 0x00, 0x8f, 0x79, 0x1f, 0x01, 0x23, 0x8a, 0x98, 0x00,
    0x01, 0x34, 0x7f, 0x00, 0x01, 0x34, 0x7f, 0x00, 0x9f, 0x78, 0x9f, 0x00, 0x7d, 0x89, 0x98, 0x00,
    0x00, 0x00, 0x0d, 0x00, 0x16, 0x09, 0x7f, 0x00, 0x69, 0x11, 0x7f, 0x00, 0x81, 0x19, 0x9f, 0x00,
    0x86, 0x39, 0x9a, 0x00, 0x00, 0x18, 0x0e, 0x00, 0x07, 0x20, 0x7f, 0x00, 0x10, 0x60, 0x7f, 0x00,
    0x01, 0x60, 0x7f, 0x00, 0x00, 0x40, 0x1f, 0x00, 0x00, 0x30, 0x7f, 0x00, 0x00, 0x28, 0x7f, 0x00,
    0xc0, 0x68, 0x7f, 0x00, 0x01, 0x20, 0x7f, 0x00, 0xaf, 0x29, 0x9f, 0x00, 0xab, 0x11, 0x9f, 0x00,
    0x9b, 0x61, 0x9f, 0x00, 0x07, 0x39, 0x6b, 0x00, 0x01, 0x50, 0x7f, 0x00, 0xa7, 0x29, 0x9f, 0x00,
    0xc9, 0x49, 0x9f, 0x00, 0x9b, 0x79, 0x9f, 0x00, 0x04, 0x24, 0x7f, 0x00, 0xb7, 0x79, 0x9f, 0x00,
    0x01, 0x34, 0x7f, 0x00, 0x9b, 0x79, 0x9f, 0x00, 0xb2, 0x31, 0x9a, 0x00, 0x48, 0x4c, 0x7f, 0x00,
    0x04, 0x24, 0x7f, 0x00, 0xcf, 0x11, 0x9f, 0x00, 0xc0, 0x31, 0x99, 0x00, 0x88, 0x4c, 0x7f, 0x00,
    0x02, 0x60, 0x7f, 0x00, 0xc0, 0x11, 0x9f, 0x00, 0x02, 0x60, 0x7f, 0x00, 0x01, 0x40, 0x7f, 0x00,
    0xc7, 0x49, 0x9f, 0x00, 0xc3, 0x79, 0x9f, 0x00, 0x08, 0x24, 0x7f, 0x00, 0x01, 0x50, 0x7f, 0x00,
    0xe9, 0x79, 0x9d, 0x00, 0x9f, 0x78, 0x9f, 0x00, 0x02, 0x34, 0x7f, 0x00, 0xb7, 0x79, 0x9f, 0x00,
    0x01, 0x50, 0x7f, 0x00, 0xda, 0x09, 0x9f, 0x00, 0x47, 0x48, 0x8e, 0x00, 0x4d, 0x78, 0x9f, 0x00,
    0x47, 0x48, 0x8d, 0x00, 0x4d, 0x78, 0x9f, 0x00, 0xe4, 0x09, 0x9f, 0x00, 0x00, 0x00, 0x0e, 0x00,
    0x01, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x01, 0x00, 0x1f, 0x00, 0x8b, 0x68, 0x7f, 0x00,
    0x40, 0x68, 0x7f, 0x00, 0xf3, 0x41, 0x9f, 0x00, 0xc1, 0x68, 0x7f, 0x00, 0x01, 0x00, 0x1f, 0x00,
    0xc2, 0x68, 0x7f, 0x00, 0x01, 0x00, 0x1f, 0x00, 0x01, 0x2c, 0x7f, 0x00, 0xfd, 0x41, 0x9f, 0x00,
    0x01, 0x00, 0x1f, 0x00, 0x05, 0x2c, 0x7f, 0x00, 0x01, 0x00, 0x1f, 0x00, 0x05, 0x39, 0x7f, 0x00,
    0x03, 0x62, 0x9f, 0x00, 0x1b, 0x0a, 0x9f, 0x00, 0x40, 0x36, 0x7f, 0x00, 0x40, 0x35, 0x7f, 0x00,
    0x01, 0x39, 0x7f, 0x00, 0x11, 0x4a, 0x99, 0x00, 0x2b, 0x4a, 0x9f, 0x00, 0x0d, 0x62, 0x9f, 0x00,
    0x17, 0x4a, 0x99, 0x00, 0x2b, 0x4a, 0x9f, 0x00, 0x1d, 0x0a, 0x9f, 0x00, 0x13, 0x7a, 0x9f, 0x00,
    0x2b, 0x4a, 0x9f, 0x00, 0x80, 0x36, 0x7f, 0x00, 0x80, 0x35, 0x7f, 0x00, 0x2b, 0x7a, 0x9f, 0x00,
    0x05, 0x39, 0x7f, 0x00, 0x01, 0x34, 0x7f, 0x00, 0x01, 0x34, 0x7f, 0x00, 0x03, 0x7a, 0x9f, 0x00,
    0x2b, 0x6a, 0x9f, 0x00, 0x01, 0x50, 0x7f, 0x00, 0x9f, 0x78, 0x9f, 0x00, 0x47, 0x48, 0x9f, 0x00,
    0x4d, 0x78, 0x9f, 0x00, 0x4d, 0x90, 0x98, 0x00, 0x3b, 0x3a, 0x9f, 0x00, 0x80, 0x1c, 0x7f, 0x00,
    0x4c, 0x00, 0x9a, 0x00, 0x4c, 0x28, 0x9f, 0x00, 0x43, 0x1a, 0x9f, 0x00, 0x08, 0x60, 0x7f, 0x00,
    0x01, 0x20, 0x7f, 0x00, 0x10, 0x24, 0x7f, 0x00, 0x4d, 0x78, 0x9f, 0x00, 0x4d, 0x8a, 0x98, 0x00,
    0x00, 0x00, 0x0d, 0x00, 0x57, 0x08, 0x7f, 0x00, 0x29, 0x10, 0x7f, 0x00, 0x55, 0x02, 0x9f, 0x00,
    0x00, 0x40, 0x1f, 0x00, 0x01, 0x60, 0x7f, 0x00, 0x53, 0x7a, 0x9f, 0x00, 0x01, 0x60, 0x7f, 0x00,
    0x57, 0x7a, 0x9f, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x9a, 0x01, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0xe0, 0x00,
    0x20, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x00, 0x07, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x00, 0xc0, 0x00, 0x00, 0x0b, 0x00, 0x00,
    0x31, 0x02, 0x02, 0x00, 0x05, 0xce, 0x00, 0x00, 0xe7, 0x60, 0x03, 0x00, 0xa8, 0xe6, 0x41, 0x09,
    0x10, 0x05, 0x10, 0x00, 0x51, 0x1e, 0xc0, 0x42, 0x01, 0x04, 0x00, 0x00, 0x00, 0x84, 0x2e, 0x00,
    0x1a, 0x58, 0x94, 0x16, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x1c, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x06, 0x1c, 0x08, 0x00, 0x10, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x06, 0x1c, 0x08, 0x00, 0x10, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x15, 0x24, 0xc0, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x40, 0x41, 0x42, 0x43, 0x44,
    0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54,
    0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63, 0x64,
    0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71
};

static uint8_t spi_fw[] __attribute__ ((aligned (4))) = {
    0x53, 0x45, 0x46, 0x57, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x30, 0x01, 0x00, 0x02, 0x0b,
    0x09, 0x00, 0xba, 0x01, 0x1c, 0x00, 0x45, 0x00, 0x18, 0x08, 0x04, 0x07, 0xce, 0x07, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa8, 0x11, 0x00,
    0x20, 0xc0, 0x34, 0x00, 0x00, 0xc8, 0x3d, 0x00, 0x10, 0x40, 0x62, 0x00, 0x00, 0xe8, 0x67, 0x00,
    0x30, 0xc0, 0x83, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xe8, 0xa8, 0x00, 0x30, 0x40, 0xd7, 0x00, 0x00, 0xe8, 0x11, 0x00,
    0x10, 0xc0, 0x33, 0x00, 0x00, 0x88, 0x8a, 0x00, 0x08, 0x40, 0x00, 0x00, 0x00, 0x08, 0x8b, 0x00,
    0x08, 0x40, 0x00, 0x00, 0x00, 0x88, 0x8c, 0x00, 0x08, 0x40, 0x90, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6b, 0x0b, 0x61, 0x00, 0x40, 0x11, 0x7f, 0x00,
    0x81, 0x38, 0x7f, 0x00, 0x83, 0x7a, 0xff, 0x00, 0x5f, 0x6f, 0x7f, 0x00, 0x07, 0x78, 0x7f, 0x00,
    0x53, 0x79, 0x7f, 0x00, 0x50, 0x78, 0x7f, 0x00, 0x9b, 0x30, 0x9f, 0x00, 0x58, 0x10, 0x9f, 0x00,
    0x61, 0x48, 0x86, 0x00, 0x04, 0x3a, 0x7f, 0x00, 0x90, 0x68, 0x9f, 0x00, 0x6f, 0x6f, 0x7f, 0x00,
    0x69, 0x48, 0x9f, 0x00, 0x1f, 0x78, 0x7f, 0x00, 0x83, 0x78, 0x9f, 0x00, 0x48, 0x78, 0x7f, 0x00,
    0x0b, 0x78, 0x7f, 0x00, 0x77, 0x78, 0x9f, 0x00, 0x6f, 0x60, 0x9f, 0x00, 0x04, 0x3a, 0x64, 0x00,
    0x72, 0x28, 0x3f, 0x01, 0x75, 0x60, 0xcc, 0x00, 0x28, 0xb8, 0x67, 0x00, 0x00, 0xba, 0x67, 0x00,
    0x7b, 0xe8, 0x87, 0x00, 0x6e, 0xb0, 0x25, 0x01, 0x9b, 0x78, 0x9f, 0x00, 0x80, 0x28, 0xc6, 0x00,
    0x28, 0xb8, 0x67, 0x00, 0x00, 0xb9, 0x67, 0x00, 0x87, 0xe0, 0x27, 0x01, 0x80, 0xb0, 0x85, 0x00,
    0x9b, 0x78, 0x9f, 0x00, 0x8c, 0x28, 0x9f, 0x00, 0x00, 0x00, 0x06, 0x00, 0x28, 0xb8, 0x67, 0x00,
    0x00, 0xba, 0x67, 0x00, 0x95, 0xe8, 0x87, 0x00, 0x91, 0xa8, 0xc5, 0x00, 0x8c, 0x30, 0x9f, 0x00,
    0xbb, 0x38, 0x3b, 0x01, 0x0b, 0x0b, 0x63, 0x00, 0x01, 0x44, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00,
    0xa3, 0x60, 0x9f, 0x00, 0xb8, 0x28, 0x83, 0x00, 0xaa, 0x28, 0x83, 0x00, 0x00, 0x1e, 0x7f, 0x00,
    0xb0, 0x28, 0x83, 0x00, 0x03, 0x39, 0x7f, 0x00, 0xaf, 0x60, 0x9f, 0x00, 0x20, 0x1c, 0x7f, 0x00,
    0xb6, 0x30, 0x9b, 0x00, 0x40, 0x1c, 0x7f, 0x00, 0x00, 0x40, 0x1f, 0x00, 0xa7, 0x78, 0x84, 0x00,
    0xc2, 0x48, 0x9f, 0x00, 0xc2, 0x30, 0x9a, 0x00, 0xbf, 0x60, 0x9f, 0x00, 0x04, 0x3a, 0x64, 0x00,
    0x01, 0x44, 0x7f, 0x00, 0x20, 0x1c, 0x7f, 0x00, 0x08, 0x20, 0x7f, 0x00, 0xc9, 0x68, 0x9f, 0x00,
    0xca, 0x08, 0x9f, 0x00, 0xff, 0x78, 0x9f, 0x00, 0xfb, 0x0b, 0x7f, 0x00, 0x89, 0x11, 0x9f, 0x00,
    0xef, 0x50, 0x9f, 0x00, 0x01, 0x44, 0x7f, 0x00, 0x01, 0x60, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00,
    0xdb, 0x60, 0x9f, 0x00, 0x4b, 0x0b, 0x72, 0x00, 0x00, 0x10, 0x7f, 0x00, 0xeb, 0x18, 0x9a, 0x00,
    0xb7, 0x08, 0x9b, 0x00, 0x01, 0x5c, 0x7f, 0x00, 0x40, 0x1c, 0x7f, 0x00, 0xb7, 0x78, 0x9f, 0x00,
    0x00, 0x1e, 0x7f, 0x00, 0xe3, 0x78, 0x9f, 0x00, 0x50, 0xf9, 0x7f, 0x00, 0xf4, 0xd0, 0x9f, 0x00,
    0xf3, 0xd0, 0x87, 0x00, 0xd5, 0x78, 0x85, 0x00, 0x00, 0x00, 0x01, 0x00, 0x83, 0x01, 0x9a, 0x00,
    0x7f, 0x09, 0x9a, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x05, 0x5c, 0x61, 0x00, 0x82, 0x38, 0x7f, 0x00,
    0x09, 0x49, 0x9f, 0x00, 0x04, 0x3a, 0x7f, 0x00, 0x28, 0x69, 0x9f, 0x00, 0x08, 0x78, 0x7f, 0x00,
    0x56, 0x79, 0x7f, 0x00, 0xef, 0x6e, 0x7f, 0x00, 0x20, 0x38, 0x7f, 0x00, 0x01, 0x2c, 0x7f, 0x00,
    0x05, 0x2c, 0x7f, 0x00, 0x05, 0x2c, 0x7f, 0x00, 0x00, 0x04, 0x7f, 0x00, 0x01, 0x40, 0x7f, 0x00,
    0x50, 0x78, 0x7f, 0x00, 0x02, 0x1c, 0x7f, 0x00, 0x1f, 0x61, 0x9f, 0x00, 0x25, 0x49, 0x9f, 0x00,
    0x45, 0x79, 0x9f, 0x00, 0x48, 0x78, 0x7f, 0x00, 0x39, 0x79, 0x9f, 0x00, 0x52, 0x79, 0x7f, 0x00,
    0xbf, 0x6e, 0x7f, 0x00, 0x2d, 0x61, 0x9f, 0x00, 0x57, 0x79, 0x9f, 0x00, 0x31, 0x61, 0x9f, 0x00,
    0x04, 0x3a, 0x64, 0x00, 0x34, 0x29, 0x3f, 0x01, 0x37, 0x61, 0x8b, 0x00, 0x28, 0x38, 0x68, 0x00,
    0x00, 0xba, 0x68, 0x00, 0x3d, 0xe9, 0x88, 0x00, 0x31, 0xd9, 0x28, 0x01, 0x5f, 0x79, 0x9f, 0x00,
    0x42, 0x29, 0x9f, 0x00, 0x28, 0x38, 0x68, 0x00, 0x00, 0xb9, 0x68, 0x00, 0x49, 0xe1, 0x28, 0x01,
    0x43, 0xd9, 0x88, 0x00, 0x5f, 0x79, 0x9f, 0x00, 0x57, 0x29, 0x88, 0x00, 0x01, 0x40, 0x7f, 0x00,
    0x52, 0x29, 0x9f, 0x00, 0x01, 0x44, 0x7f, 0x00, 0x20, 0x38, 0x68, 0x00, 0x59, 0xd1, 0x88, 0x00,
    0x4f, 0x59, 0x88, 0x00, 0x5f, 0x79, 0x88, 0x00, 0x0b, 0x0b, 0x7f, 0x00, 0x01, 0x44, 0x7f, 0x00,
    0x05, 0x39, 0x7f, 0x00, 0x65, 0x61, 0x9f, 0x00, 0x7c, 0x29, 0x83, 0x00, 0x6c, 0x29, 0x83, 0x00,
    0x00, 0x1e, 0x7f, 0x00, 0x72, 0x29, 0x83, 0x00, 0x03, 0x39, 0x7f, 0x00, 0x71, 0x61, 0x9f, 0x00,
    0x76, 0x31, 0x9b, 0x00, 0x20, 0x1c, 0x7f, 0x00, 0x09, 0x20, 0x7f, 0x00, 0x40, 0x1c, 0x7f, 0x00,
    0x00, 0x40, 0x1f, 0x00, 0x69, 0x79, 0x84, 0x00, 0x00, 0x1e, 0x7f, 0x00, 0xfd, 0x78, 0x9f, 0x00,
    0x03, 0x39, 0x7f, 0x00, 0x85, 0x61, 0x9f, 0x00, 0xfb, 0x78, 0x9f, 0x00, 0x89, 0xd1, 0xdf, 0x00,
    0x8f, 0x11, 0x9b, 0x00, 0x01, 0x5c, 0x7f, 0x00, 0x01, 0x60, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00,
    0x01, 0x44, 0x7f, 0x00, 0x95, 0x61, 0x9f, 0x00, 0x9b, 0x19, 0x9a, 0x00, 0x73, 0x79, 0x92, 0x00,
    0x00, 0x1e, 0x72, 0x00, 0x73, 0x79, 0x9f, 0x00, 0x5f, 0x6e, 0x61, 0x00, 0x6b, 0x0b, 0x7f, 0x00,
    0x40, 0x11, 0x7f, 0x00, 0x05, 0x5c, 0x7f, 0x00, 0x81, 0x38, 0x7f, 0x00, 0x83, 0x7a, 0xff, 0x00,
    0x53, 0x79, 0x7f, 0x00, 0x09, 0x78, 0x7f, 0x00, 0x50, 0x78, 0x7f, 0x00, 0xb7, 0x49, 0x86, 0x00,
    0x04, 0x3a, 0x7f, 0x00, 0xe6, 0x69, 0x9f, 0x00, 0x6f, 0x6e, 0x7f, 0x00, 0xbf, 0x49, 0x9f, 0x00,
    0x1f, 0x78, 0x7f, 0x00, 0xd9, 0x79, 0x9f, 0x00, 0x48, 0x78, 0x7f, 0x00, 0x0b, 0x78, 0x7f, 0x00,
    0xcd, 0x79, 0x9f, 0x00, 0xc5, 0x61, 0x9f, 0x00, 0x04, 0x3a, 0x64, 0x00, 0xc8, 0x29, 0x3f, 0x01,
    0xcb, 0x61, 0xcc, 0x00, 0x28, 0xb8, 0x69, 0x00, 0x00, 0xba, 0x69, 0x00, 0xd1, 0xe9, 0x89, 0x00,
    0xc4, 0xb1, 0x28, 0x01, 0xf1, 0x79, 0x9f, 0x00, 0xd6, 0x29, 0xc6, 0x00, 0x28, 0xb8, 0x69, 0x00,
    0x00, 0xb9, 0x69, 0x00, 0xdd, 0xe1, 0x29, 0x01, 0xd6, 0xb1, 0x88, 0x00, 0xf1, 0x79, 0x9f, 0x00,
    0xe2, 0x29, 0x9f, 0x00, 0x00, 0x00, 0x06, 0x00, 0x28, 0xb8, 0x69, 0x00, 0x00, 0xba, 0x69, 0x00,
    0xeb, 0xe9, 0x89, 0x00, 0xe7, 0xa9, 0xc8, 0x00, 0xe2, 0x31, 0x9f, 0x00, 0x0b, 0x0b, 0x63, 0x00,
    0x01, 0x44, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0xf7, 0x61, 0x9f, 0x00, 0x0c, 0x2a, 0x83, 0x00,
    0x00, 0x00, 0x1f, 0x00, 0x00, 0x2a, 0x83, 0x00, 0x00, 0x1e, 0x7f, 0x00, 0x06, 0x2a, 0x83, 0x00,
    0x03, 0x39, 0x7f, 0x00, 0x05, 0x62, 0x9f, 0x00, 0x20, 0x1c, 0x7f, 0x00, 0x40, 0x1c, 0x7f, 0x00,
    0x00, 0x40, 0x1f, 0x00, 0xfd, 0x79, 0x84, 0x00, 0x23, 0x52, 0x9f, 0x00, 0x01, 0x44, 0x7f, 0x00,
    0x01, 0x60, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x17, 0x62, 0x9f, 0x00, 0x08, 0x5c, 0x72, 0x00,
    0x1e, 0x1a, 0x9a, 0x00, 0x00, 0x1e, 0x7f, 0x00, 0x40, 0x1c, 0x7f, 0x00, 0x00, 0x40, 0x1f, 0x00,
    0x50, 0xf9, 0x7f, 0x00, 0x28, 0xd2, 0x9f, 0x00, 0x27, 0xd2, 0x89, 0x00, 0x11, 0x7a, 0x88, 0x00,
    0x00, 0x40, 0x0b, 0x00, 0x30, 0x02, 0x9f, 0x00, 0x01, 0x60, 0x7f, 0x00, 0x00, 0x40, 0x04, 0x00,
    0x38, 0x3a, 0x9a, 0x00, 0x01, 0x40, 0x6f, 0x00, 0x37, 0x7a, 0x8f, 0x00, 0x20, 0x38, 0x6f, 0x00,
    0x00, 0x00, 0x0f, 0x00, 0x3b, 0xd2, 0x8f, 0x00, 0x00, 0x20, 0x0f, 0x00, 0x00, 0x40, 0x1f, 0x00,
    0x43, 0x62, 0x9f, 0x00, 0x7f, 0x6d, 0x7f, 0x00, 0x6b, 0x09, 0x7f, 0x00, 0xc4, 0x78, 0x7f, 0x00,
    0x07, 0x78, 0x7f, 0x00, 0x04, 0x3a, 0x7f, 0x00, 0x56, 0x6a, 0x9f, 0x00, 0x01, 0x20, 0x7f, 0x00,
    0x57, 0x7a, 0x9f, 0x00, 0x61, 0x2a, 0xc5, 0x00, 0xd5, 0x00, 0x9f, 0x00, 0x56, 0x12, 0x9f, 0x00,
    0x04, 0x3a, 0x7f, 0x00, 0x5d, 0x6a, 0x9f, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0xb9, 0x67, 0x00,
    0x82, 0x1c, 0x67, 0x00, 0x65, 0x62, 0x87, 0x00, 0x55, 0x52, 0x87, 0x00, 0x00, 0x00, 0x05, 0x00,
    0xd5, 0x00, 0x9f, 0x00, 0x00, 0x10, 0x1f, 0x00, 0x79, 0x4a, 0x9f, 0x00, 0x04, 0x3a, 0x7f, 0x00,
    0x73, 0x6a, 0x9f, 0x00, 0x00, 0x30, 0x05, 0x00, 0x6b, 0x7a, 0x85, 0x00, 0x6b, 0x0b, 0x7f, 0x00,
    0x20, 0x38, 0x7f, 0x00, 0xcc, 0x78, 0x7f, 0x00, 0x9b, 0x38, 0x9b, 0x00, 0x95, 0x7a, 0x9f, 0x00,
    0x88, 0x02, 0x9a, 0x00, 0x03, 0x39, 0x7f, 0x00, 0x87, 0x62, 0x9f, 0x00, 0x8c, 0x0a, 0x9a, 0x00,
    0x00, 0x1e, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x93, 0x3a, 0x9b, 0x00, 0x94, 0x0a, 0x9b, 0x00,
    0x43, 0x2a, 0x9a, 0x00, 0x01, 0x2c, 0x7f, 0x00, 0x05, 0x2c, 0x7f, 0x00, 0x05, 0x2c, 0x7f, 0x00,
    0x00, 0x04, 0x7f, 0x00, 0x01, 0x40, 0x7f, 0x00, 0x9f, 0x62, 0x9f, 0x00, 0x01, 0x00, 0x1f, 0x00,
    0x6b, 0x0b, 0x61, 0x00, 0x40, 0x11, 0x7f, 0x00, 0x05, 0x5c, 0x7f, 0x00, 0x82, 0x38, 0x7f, 0x00,
    0x83, 0x7a, 0xff, 0x00, 0x01, 0x44, 0x7f, 0x00, 0x53, 0x79, 0x7f, 0x00, 0x5f, 0x6e, 0x7f, 0x00,
    0x09, 0x78, 0x7f, 0x00, 0x50, 0x78, 0x7f, 0x00, 0xbd, 0x4a, 0x86, 0x00, 0x04, 0x3a, 0x7f, 0x00,
    0x24, 0x6b, 0x9f, 0x00, 0x6f, 0x6e, 0x7f, 0x00, 0xc5, 0x4a, 0x9f, 0x00, 0x1f, 0x78, 0x7f, 0x00,
    0xfb, 0x7a, 0x9f, 0x00, 0x48, 0x78, 0x7f, 0x00, 0x0b, 0x78, 0x7f, 0x00, 0xd5, 0x7a, 0x9f, 0x00,
    0xcb, 0x62, 0x9f, 0x00, 0x04, 0x3a, 0x64, 0x00, 0x5d, 0x03, 0x9f, 0x00, 0xce, 0x2a, 0x3f, 0x01,
    0xd3, 0x62, 0xcc, 0x00, 0x20, 0x38, 0x69, 0x00, 0xd7, 0xd2, 0x89, 0x00, 0xea, 0x5a, 0x89, 0x00,
    0xca, 0x32, 0x28, 0x01, 0xef, 0x6e, 0x7f, 0x00, 0x08, 0x20, 0x7f, 0x00, 0x08, 0x78, 0x7f, 0x00,
    0x02, 0x1c, 0x7f, 0x00, 0x56, 0x79, 0x7f, 0x00, 0x01, 0x40, 0x7f, 0x00, 0x31, 0x79, 0x83, 0x00,
    0x6f, 0x6f, 0x68, 0x00, 0x0b, 0x78, 0x7f, 0x00, 0x04, 0x20, 0x7f, 0x00, 0x01, 0x40, 0x7f, 0x00,
    0x6e, 0x30, 0x3f, 0x01, 0x43, 0x7b, 0x9f, 0x00, 0x5d, 0x03, 0x9f, 0x00, 0xf6, 0x2a, 0xc6, 0x00,
    0x20, 0x38, 0x69, 0x00, 0xfd, 0xd2, 0x29, 0x01, 0x12, 0x5b, 0x89, 0x00, 0xf8, 0x32, 0x88, 0x00,
    0xef, 0x6e, 0x7f, 0x00, 0x08, 0x20, 0x7f, 0x00, 0x08, 0x78, 0x7f, 0x00, 0x02, 0x1c, 0x7f, 0x00,
    0x56, 0x79, 0x7f, 0x00, 0x01, 0x40, 0x7f, 0x00, 0x45, 0x29, 0x83, 0x00, 0x43, 0x79, 0x9f, 0x00,
    0x6f, 0x6f, 0x68, 0x00, 0x1f, 0x78, 0x7f, 0x00, 0x04, 0x20, 0x7f, 0x00, 0x01, 0x40, 0x7f, 0x00,
    0x80, 0x30, 0x9f, 0x00, 0x43, 0x7b, 0x9f, 0x00, 0x5d, 0x03, 0x9f, 0x00, 0x1e, 0x2b, 0x9f, 0x00,
    0x00, 0x00, 0x06, 0x00, 0x20, 0x38, 0x69, 0x00, 0x27, 0xd3, 0x89, 0x00, 0x36, 0x5b, 0x89, 0x00,
    0x25, 0x2b, 0xc8, 0x00, 0x1e, 0x33, 0x9f, 0x00, 0x52, 0x79, 0x7f, 0x00, 0xbf, 0x6e, 0x63, 0x00,
    0x57, 0x29, 0x9f, 0x00, 0x51, 0x79, 0x9f, 0x00, 0x5f, 0x6f, 0x68, 0x00, 0x07, 0x78, 0x7f, 0x00,
    0x04, 0x20, 0x7f, 0x00, 0x01, 0x40, 0x7f, 0x00, 0x8f, 0x28, 0x9f, 0x00, 0x8c, 0x30, 0x9f, 0x00,
    0x0b, 0x0b, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x47, 0x63, 0x9f, 0x00, 0x4d, 0x13, 0x9a, 0x00,
    0x00, 0x00, 0x04, 0x00, 0x50, 0x1b, 0x9a, 0x00, 0x00, 0x1e, 0x7f, 0x00, 0x56, 0x23, 0x9a, 0x00,
    0x03, 0x39, 0x7f, 0x00, 0x55, 0x63, 0x9f, 0x00, 0x20, 0x1c, 0x7f, 0x00, 0x40, 0x1c, 0x7f, 0x00,
    0x00, 0x40, 0x1f, 0x00, 0x62, 0x53, 0x9f, 0x00, 0x89, 0x91, 0x98, 0x00, 0xef, 0xb0, 0x98, 0x00,
    0x01, 0x44, 0x7f, 0x00, 0x01, 0x60, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x69, 0x63, 0x9f, 0x00,
    0x08, 0x5c, 0x72, 0x00, 0x70, 0x1b, 0x9a, 0x00, 0x00, 0x1e, 0x7f, 0x00, 0x40, 0x1c, 0x7f, 0x00,
    0x00, 0x40, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x78, 0x0e, 0x24, 0x00, 0x88, 0x10, 0x01, 0x00, 0x07, 0x00, 0x24, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x15, 0x00,
    0xa0, 0x83, 0x14, 0x00, 0x28, 0x81, 0xaa, 0x00, 0x02, 0x10, 0x64, 0x00, 0x00, 0x40, 0x00, 0x00,
    0x01, 0x02, 0x00, 0x00, 0x05, 0xfe, 0x01, 0x00, 0xe7, 0xc2, 0x02, 0x00, 0x00, 0x5c, 0x43, 0x0a,
    0x1a, 0x01, 0x10, 0x00, 0x00, 0x00, 0x80, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0xac, 0x0c, 0x10,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xe4, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x07, 0xf8, 0x07, 0x00, 0xfe, 0xfe, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x09, 0x00, 0x00, 0x00, 0x07, 0xf8, 0x07, 0x00, 0xfe, 0xfe, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x3f, 0x03, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x40, 0x41, 0x42, 0x43, 0x44,
    0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54,
    0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63, 0x64,
    0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71
};

static uint8_t i2c_fw[] __attribute__ ((aligned (4))) = {
    0x53, 0x45, 0x46, 0x57, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x30, 0x03, 0x00, 0x04, 0x02,
    0x09, 0x00, 0x57, 0x01, 0x1c, 0x00, 0x45, 0x00, 0x8c, 0x06, 0x78, 0x05, 0xde, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x0a, 0x00,
    0x20, 0x6d, 0x59, 0x00, 0x00, 0x48, 0x0a, 0x00, 0x10, 0x6d, 0x59, 0x00, 0x00, 0x68, 0x0a, 0x00,
    0x10, 0x6d, 0x59, 0x00, 0x00, 0x08, 0x0a, 0x00, 0x28, 0x6d, 0x59, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0xa4, 0x00, 0x08, 0x6d, 0x59, 0x00, 0x00, 0x08, 0x9b, 0x00,
    0x08, 0x42, 0x59, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x78, 0x7f, 0x00,
    0x1f, 0x78, 0x7f, 0x00, 0xe7, 0x6f, 0x7f, 0x00, 0x20, 0x08, 0x7f, 0x00, 0x10, 0x4c, 0x7f, 0x00,
    0xfb, 0x11, 0x9b, 0x00, 0x73, 0x41, 0x9a, 0x00, 0x00, 0x00, 0x01, 0x00, 0x3d, 0x48, 0x99, 0x00,
    0x04, 0x20, 0x7f, 0x00, 0x45, 0x20, 0x9b, 0x00, 0x45, 0x08, 0x9b, 0x00, 0x44, 0x38, 0x9f, 0x00,
    0xdd, 0x79, 0x9f, 0x00, 0xa3, 0x01, 0x9c, 0x00, 0x00, 0x00, 0x09, 0x00, 0x81, 0x38, 0x7f, 0x00,
    0xa9, 0x09, 0x9c, 0x00, 0x00, 0x00, 0x09, 0x00, 0x28, 0x08, 0x7f, 0x00, 0x00, 0x10, 0x7f, 0x00,
    0x5b, 0x38, 0x98, 0x00, 0x5b, 0xb0, 0x98, 0x00, 0x5a, 0x38, 0x99, 0x00, 0x4f, 0x78, 0x9f, 0x00,
    0x5a, 0x18, 0x9f, 0x00, 0xe8, 0x33, 0x3f, 0x00, 0x65, 0x38, 0x98, 0x00, 0x65, 0xb0, 0x98, 0x00,
    0x4f, 0x38, 0x99, 0x00, 0xfe, 0x19, 0x98, 0x00, 0x07, 0x04, 0x65, 0x00, 0xb0, 0x19, 0x98, 0x00,
    0xb0, 0x19, 0x98, 0x00, 0xb0, 0x19, 0x98, 0x00, 0xb0, 0x19, 0x98, 0x00, 0xb0, 0x19, 0x98, 0x00,
    0xb0, 0x19, 0x98, 0x00, 0xb0, 0x19, 0x98, 0x00, 0x06, 0x39, 0x7f, 0x00, 0x78, 0x28, 0xc7, 0x00,
    0x04, 0x3a, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x81, 0x60, 0xc7, 0x00,
    0x06, 0x39, 0x7f, 0x00, 0x82, 0x18, 0x9f, 0x00, 0x64, 0x89, 0x9f, 0x00, 0x79, 0x50, 0x9f, 0x00,
    0x03, 0x12, 0x9b, 0x00, 0x90, 0x38, 0x9f, 0x00, 0x03, 0x1a, 0x9b, 0x00, 0x90, 0x28, 0xc7, 0x00,
    0x04, 0x3a, 0x69, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x01, 0x20, 0x7f, 0x00,
    0x9b, 0x60, 0xc7, 0x00, 0x06, 0x39, 0x7f, 0x00, 0x9c, 0x18, 0x9f, 0x00, 0x64, 0x09, 0x9f, 0x00,
    0xa2, 0x28, 0xc7, 0x00, 0x04, 0x3a, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0xb1, 0x30, 0x9a, 0x00,
    0xaf, 0x48, 0x9f, 0x00, 0xb1, 0x78, 0x9f, 0x00, 0x02, 0x24, 0x79, 0x00, 0x00, 0x00, 0x12, 0x00,
    0x8d, 0x21, 0x9b, 0x00, 0xb5, 0x60, 0xc7, 0x00, 0x06, 0x39, 0x7f, 0x00, 0xb6, 0x18, 0x9f, 0x00,
    0xbf, 0x30, 0x9a, 0x00, 0x79, 0x49, 0x82, 0x00, 0x08, 0x04, 0x7f, 0x00, 0xc0, 0x28, 0xc7, 0x00,
    0x04, 0x3a, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x00, 0x80, 0x13, 0x00, 0xc9, 0x60, 0xc7, 0x00,
    0x06, 0x39, 0x7f, 0x00, 0xca, 0x18, 0x9f, 0x00, 0x64, 0x09, 0x9f, 0x00, 0xc1, 0x50, 0x9f, 0x00,
    0xd2, 0x28, 0xc7, 0x00, 0x04, 0x3a, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x08, 0x04, 0x72, 0x00,
    0x80, 0x1c, 0x7f, 0x00, 0x28, 0x08, 0x7f, 0x00, 0xd7, 0x71, 0x9f, 0x00, 0x02, 0x1c, 0x7f, 0x00,
    0x87, 0x59, 0x9f, 0x00, 0xed, 0x78, 0x83, 0x00, 0x8c, 0x21, 0x9f, 0x00, 0xec, 0x10, 0x9f, 0x00,
    0x89, 0x79, 0x9f, 0x00, 0xed, 0x60, 0xc7, 0x00, 0x06, 0x39, 0x7f, 0x00, 0xee, 0x18, 0x9f, 0x00,
    0xf7, 0x30, 0x9a, 0x00, 0x79, 0x49, 0x9f, 0x00, 0xf6, 0x28, 0xc7, 0x00, 0x04, 0x3a, 0x7f, 0x00,
    0x05, 0x39, 0x7f, 0x00, 0x20, 0x08, 0x7f, 0x00, 0xc1, 0x01, 0x99, 0x00, 0x09, 0x39, 0x9a, 0x00,
    0x1a, 0x39, 0x9f, 0x00, 0x1b, 0x09, 0x9b, 0x00, 0xdd, 0x79, 0x9f, 0x00, 0x00, 0x00, 0x1f, 0x00,
    0x06, 0x39, 0x7f, 0x00, 0x0a, 0x19, 0x9f, 0x00, 0x12, 0x19, 0x98, 0x00, 0x0e, 0x29, 0x9f, 0x00,
    0x04, 0x3a, 0x67, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x00, 0x00, 0x12, 0x00,
    0x35, 0x11, 0x9a, 0x00, 0xec, 0x03, 0x25, 0x00, 0xad, 0x79, 0x9d, 0x00, 0xf3, 0x79, 0x9d, 0x00,
    0x07, 0x2e, 0x7f, 0x00, 0x08, 0x39, 0x7f, 0x00, 0x62, 0x19, 0x98, 0x00, 0x27, 0x61, 0x9f, 0x00,
    0x62, 0x19, 0x98, 0x00, 0x00, 0x00, 0x12, 0x00, 0x64, 0x19, 0x98, 0x00, 0x64, 0x19, 0x98, 0x00,
    0x39, 0x79, 0x9f, 0x00, 0xe4, 0x03, 0x29, 0x00, 0x40, 0x32, 0x3f, 0x00, 0xa9, 0x19, 0x9c, 0x00,
    0xa3, 0x21, 0x9c, 0x00, 0x4c, 0x39, 0x9f, 0x00, 0x49, 0x41, 0x9f, 0x00, 0x4f, 0x21, 0x9b, 0x00,
    0x21, 0x08, 0x7f, 0x00, 0x55, 0x01, 0x9f, 0x00, 0x20, 0x1c, 0x7f, 0x00, 0x4d, 0x19, 0x9b, 0x00,
    0x4f, 0x79, 0x9f, 0x00, 0x40, 0x1c, 0x7f, 0x00, 0x5d, 0x11, 0x9a, 0x00, 0x05, 0x20, 0x7f, 0x00,
    0x00, 0x40, 0x1f, 0x00, 0x01, 0x60, 0x7f, 0x00, 0x4f, 0x09, 0x9b, 0x00, 0x01, 0x5c, 0x7f, 0x00,
    0x4d, 0x79, 0x9f, 0x00, 0x51, 0xc1, 0x99, 0x00, 0x04, 0x20, 0x7f, 0x00, 0x53, 0x11, 0x9a, 0x00,
    0x00, 0x00, 0x12, 0x00, 0x10, 0x24, 0x79, 0x00, 0x6f, 0x11, 0x9b, 0x00, 0x6f, 0x19, 0x9b, 0x00,
    0x05, 0x20, 0x7f, 0x00, 0x00, 0x40, 0x1f, 0x00, 0x40, 0x1c, 0x7f, 0x00, 0x6b, 0x79, 0x9f, 0x00,
    0x82, 0x38, 0x7f, 0x00, 0x07, 0x04, 0x7f, 0x00, 0x2f, 0x7a, 0x9f, 0x00, 0xf7, 0x38, 0x9a, 0x00,
    0x9f, 0x21, 0x9b, 0x00, 0x9f, 0x29, 0x9a, 0x00, 0x50, 0x4c, 0x7f, 0x00, 0x02, 0x24, 0x79, 0x00,
    0x03, 0x20, 0x7f, 0x00, 0xc7, 0x79, 0x9f, 0x00, 0xd1, 0x31, 0x9f, 0x00, 0x8d, 0x01, 0x99, 0x00,
    0x88, 0x11, 0x98, 0x00, 0x8d, 0x61, 0xc7, 0x00, 0x28, 0x08, 0x7f, 0x00, 0x06, 0x39, 0x7f, 0x00,
    0x90, 0x19, 0xd2, 0x00, 0xf7, 0x00, 0x99, 0x00, 0x9b, 0x31, 0x9a, 0x00, 0x79, 0x49, 0x9f, 0x00,
    0xbe, 0x20, 0x9b, 0x00, 0xc7, 0x79, 0x9f, 0x00, 0x02, 0x24, 0x7f, 0x00, 0x83, 0x79, 0x9f, 0x00,
    0x03, 0x39, 0x7f, 0x00, 0xa5, 0x61, 0x9f, 0x00, 0x01, 0x00, 0x1f, 0x00, 0x00, 0x1e, 0x7f, 0x00,
    0x01, 0x00, 0x1f, 0x00, 0xad, 0x61, 0xc7, 0x00, 0x01, 0x00, 0x1f, 0x00, 0x04, 0x3a, 0x67, 0x00,
    0x00, 0x00, 0x1f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x7f, 0x78, 0x9f, 0x00, 0xb8, 0x29, 0xc7, 0x00,
    0x04, 0x3a, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x01, 0x00, 0x1f, 0x00, 0x01, 0x60, 0x7f, 0x00,
    0x29, 0x08, 0x7f, 0x00, 0x1d, 0x79, 0x89, 0x00, 0xc6, 0x29, 0xc7, 0x00, 0x04, 0x3a, 0x7f, 0x00,
    0x05, 0x39, 0x7f, 0x00, 0x29, 0x08, 0x7f, 0x00, 0x1d, 0x79, 0x89, 0x00, 0x88, 0x21, 0x9f, 0x00,
    0xec, 0x10, 0x9f, 0x00, 0x89, 0x79, 0x9f, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x87, 0x79, 0x9f, 0x00,
    0xdb, 0x01, 0x9f, 0x00, 0x05, 0x5c, 0x7f, 0x00, 0x20, 0x1c, 0x7f, 0x00, 0xb4, 0x3c, 0x72, 0x00,
    0x00, 0x08, 0x1f, 0x00, 0xe5, 0x61, 0x9f, 0x00, 0xe4, 0x03, 0x27, 0x00, 0x00, 0x08, 0x09, 0x00,
    0x4f, 0x78, 0x9f, 0x00, 0x06, 0x39, 0x7f, 0x00, 0xec, 0x19, 0x9f, 0x00, 0x01, 0x00, 0x1f, 0x00,
    0x06, 0x39, 0x7f, 0x00, 0xf2, 0x19, 0x9f, 0x00, 0x64, 0x09, 0x9f, 0x00, 0x01, 0x00, 0x1f, 0x00,
    0x05, 0x5c, 0x7f, 0x00, 0x35, 0x78, 0x9f, 0x00, 0xfe, 0x39, 0x99, 0x00, 0x5b, 0x78, 0x9f, 0x00,
    0x02, 0x2a, 0xc7, 0x00, 0x04, 0x3a, 0x7f, 0x00, 0x05, 0x39, 0x69, 0x00, 0x00, 0x00, 0x12, 0x00,
    0xad, 0x79, 0x9d, 0x00, 0xed, 0x79, 0x9d, 0x00, 0x13, 0x4a, 0x89, 0x00, 0x65, 0x79, 0x9f, 0x00,
    0x82, 0x38, 0x7f, 0x00, 0x08, 0x04, 0x7f, 0x00, 0xb9, 0x79, 0x9d, 0x00, 0xad, 0x79, 0x9d, 0x00,
    0xed, 0x79, 0x9d, 0x00, 0x21, 0x32, 0x9a, 0x00, 0x79, 0x49, 0x9f, 0x00, 0x27, 0x5a, 0x9f, 0x00,
    0x03, 0x20, 0x7f, 0x00, 0x09, 0x79, 0x9f, 0x00, 0x26, 0x2a, 0xc7, 0x00, 0x04, 0x3a, 0x7f, 0x00,
    0x05, 0x39, 0x7f, 0x00, 0x00, 0x80, 0x12, 0x00, 0x2f, 0x62, 0xc7, 0x00, 0x06, 0x39, 0x7f, 0x00,
    0x30, 0x1a, 0x9f, 0x00, 0x27, 0x52, 0x88, 0x00, 0x08, 0x04, 0x7f, 0x00, 0x38, 0x2a, 0xc7, 0x00,
    0x04, 0x3a, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x02, 0x1c, 0x7f, 0x00, 0x47, 0x02, 0x99, 0x00,
    0x5d, 0x5a, 0x9f, 0x00, 0x4d, 0x3a, 0x9a, 0x00, 0x28, 0x08, 0x7f, 0x00, 0x00, 0x00, 0x12, 0x00,
    0x51, 0xfa, 0x9f, 0x00, 0x00, 0x00, 0x05, 0x00, 0xf7, 0xf8, 0x9f, 0x00, 0x51, 0x62, 0xc7, 0x00,
    0x06, 0x39, 0x7f, 0x00, 0x52, 0x1a, 0x9f, 0x00, 0x64, 0x09, 0x9f, 0x00, 0x02, 0x20, 0x7f, 0x00,
    0xf7, 0x48, 0x82, 0x00, 0x23, 0x08, 0x65, 0x00, 0x5e, 0x0a, 0x9f, 0x00, 0x00, 0x08, 0x1f, 0x00,
    0x28, 0x08, 0x7f, 0x00, 0x65, 0x62, 0xc7, 0x00, 0x06, 0x39, 0x7f, 0x00, 0x66, 0x1a, 0x9f, 0x00,
    0x27, 0x7a, 0x9f, 0x00, 0x21, 0x08, 0x7f, 0x00, 0x72, 0x1a, 0x9f, 0x00, 0x7d, 0x7a, 0x9f, 0x00,
    0xec, 0x23, 0x25, 0x00, 0xad, 0x79, 0x9d, 0x00, 0x76, 0x1a, 0x9f, 0x00, 0xe8, 0x33, 0x3f, 0x00,
    0xf8, 0x03, 0x32, 0x00, 0x00, 0x18, 0x1f, 0x00, 0x83, 0x4a, 0x9f, 0x00, 0x8b, 0x7a, 0x9d, 0x00,
    0x00, 0x10, 0x1f, 0x00, 0x05, 0x20, 0x7f, 0x00, 0x8f, 0x02, 0x9f, 0x00, 0x00, 0x40, 0x1f, 0x00,
    0x08, 0x24, 0x79, 0x00, 0x01, 0x00, 0x1f, 0x00, 0x01, 0x60, 0x7f, 0x00, 0x00, 0x40, 0x1f, 0x00,
    0x20, 0x08, 0x7f, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x0a, 0x04, 0x7f, 0x00, 0xaa, 0x1a, 0x9f, 0x00,
    0xad, 0x79, 0x9d, 0x00, 0x00, 0x00, 0x09, 0x00, 0x9e, 0x1a, 0x9f, 0x00, 0x06, 0x39, 0x7f, 0x00,
    0xb9, 0x79, 0x9d, 0x00, 0x00, 0x80, 0x02, 0x00, 0x9b, 0x52, 0x9f, 0x00, 0x00, 0x40, 0x1f, 0x00,
    0x09, 0x04, 0x7f, 0x00, 0x9b, 0x7a, 0x9f, 0x00, 0x90, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x90, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x80, 0x03, 0x00,
    0x80, 0x40, 0x08, 0x00, 0x43, 0x03, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x1a, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x08, 0x80, 0x80, 0x00, 0x20, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x00, 0x00, 0x01, 0xfc, 0x01, 0x00,
    0x22, 0x62, 0x03, 0x00, 0xfc, 0x1f, 0xc0, 0x09, 0x20, 0x01, 0x10, 0x00, 0x00, 0x00, 0xc0, 0x02,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x04, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xf8, 0x07, 0x00, 0xfe, 0xfe, 0x0f, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x07, 0xf8, 0x07, 0x00,
    0xfe, 0xfe, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x55, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
    0x10, 0x11, 0x12, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c,
    0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c,
    0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c,
    0x6d, 0x6e, 0x6f, 0x70, 0x71
};

/**
 * geni_se_read_proto() - Read the protocol configured for a serial engine
 * @se_base:		Base address of SE.
 *
 * Return: Protocol value as configured in the serial engine.
 */
static u32 geni_se_read_proto(phys_addr_t se_base)
{
	u32 val;

	val = readl(se_base + GENI_FW_REVISION_RO);

	return (val & FW_REV_PROTOCOL_MSK) >> FW_REV_PROTOCOL_SHFT;
}

int geni_se_fw_load(phys_addr_t se_base, uint8_t se_mode)
{
	se_fw_hdr *hdr;
	uint8_t *fw;
	uint32_t *fw32;
	uint32_t i, val, cfg_val_off, cfg_indx_off, cfg_limit;

	if (se_mode == QUPV3_SE_UART) {
		hdr = (se_fw_hdr*)&uart_fw;
		fw = (uint8_t*)&uart_fw;
	} else if (se_mode == QUPV3_SE_SPI) {
		hdr = (se_fw_hdr*)&spi_fw;
		fw = (uint8_t*)&spi_fw;
	} else if (se_mode == QUPV3_SE_I2C) {
		hdr = (se_fw_hdr*)&i2c_fw;
		fw = (uint8_t*)&i2c_fw;
	}

	/* Disable Peripheral clock and Output control */
	writel(0x0, se_base + SE_GENI_OUTPUT_CTRL_OFFSET);
	writel(0x378, se_base + SE_GENI_CGC_CTRL_OFFSET);
	writel(0x0, se_base + SE_GENI_CLK_CTRL_OFFSET);
	writel(0x78, se_base + SE_GENI_CGC_CTRL_OFFSET);

	/* Configure INIT_CFG revision */
	val = *((uint16_t*)&hdr->config_version);
	writel(val, se_base + SE_GENI_INIT_CFG_REVISION_OFFSET);

	/* CFG_REG Programming */
	cfg_indx_off = hdr->config_idx_offset;
	cfg_val_off = *((uint16_t*)&hdr->config_val_offset);
	fw32 = (uint32_t*)&fw[cfg_val_off];
	cfg_limit = *((uint16_t*)&hdr->config_sz_in_items);
	for (i=0; i<cfg_limit; i++) {
		writel(fw32[i], (se_base + SE_GENI4_IMAGE_REG_OFFSET +
				(fw[cfg_indx_off+i] * 4)));
	}

	/* If SE is not using mini-core, program CFG_RAMn region as well */
	val = readl(se_base + SE_HW_PARAM_2_OFFSET);
	if (!(val & SE_GENI_USE_MINICORES_MASK)) {
		cfg_val_off = *((uint16_t*)&hdr->firmware_offset);
		fw32 = (uint32_t*)&fw[cfg_val_off];
		cfg_limit = *((uint16_t*)&hdr->firmware_sz_in_items);

		for (i=0; i<cfg_limit; i++) {
			writel(fw32[i], (se_base + SE_GENI_CFG_RAMn_OFFSET +
						(i * 4)));
		}
	}

	/* Enable Peripheral clock */
	writel(0x1, se_base + SE_GENI_FORCE_DEFAULT_REG_OFFSET);
	writel(0x378, se_base + SE_GENI_CGC_CTRL_OFFSET);
	writel(0x1, se_base + SE_GENI_CLK_CTRL_OFFSET);
	writel(0x78, se_base + SE_GENI_CGC_CTRL_OFFSET);

	/* Configure FW Revision and Protocol */
	val = ((*((uint16_t*)&hdr->serial_protocol) << 8) |
			(*((uint16_t*)&hdr->firmware_version) >> 8));
	writel(val, se_base + SE_GENI_FW_REVISION_OFFSET);
	val = (*((uint16_t*)&hdr->firmware_version) & 0xFF);
	writel(val, se_base + SE_GENI_S_FW_REVISION_OFFSET);

	/* Enable Output control */
	writel(0x7F, se_base + SE_GENI_OUTPUT_CTRL_OFFSET);

	/* Enable DMA for SPI SE alone */
	if (se_mode == QUPV3_SE_SPI)
		writel(0x1, se_base + SE_DMA_IF_EN_OFFSET);

	if (geni_se_read_proto(se_base) != se_mode)
		return -EPROTONOSUPPORT;

	return 0;
}
#endif /* CONFIG_QCOM_GENI_SE_FW_LOAD */

void geni_se_io_set_mode(phys_addr_t base)
{
	u32 val;

	val = readl(base + SE_IRQ_EN);
	val |= GENI_M_IRQ_EN | GENI_S_IRQ_EN;
	val |= DMA_TX_IRQ_EN | DMA_RX_IRQ_EN;
	writel(val, base + SE_IRQ_EN);

	val = readl(base + SE_GENI_DMA_MODE_EN);
	val &= ~GENI_DMA_MODE_EN;
	writel(val, base + SE_GENI_DMA_MODE_EN);

	writel(0, base + SE_GSI_EVENT_EN);
}

void geni_se_irq_clear(phys_addr_t base)
{
	writel(0xffffffff, base + SE_GENI_M_IRQ_CLEAR);
	writel(0xffffffff, base + SE_GENI_S_IRQ_CLEAR);
	writel(0xffffffff, base + SE_DMA_TX_IRQ_CLR);
	writel(0xffffffff, base + SE_DMA_RX_IRQ_CLR);
	writel(0xffffffff, base + SE_IRQ_EN);
}

void geni_se_config_fifo_mode(phys_addr_t base)
{
	u32 val, val_old;

	geni_se_irq_clear(base);

	val_old = val = readl(base + SE_GENI_M_IRQ_EN);
	val |= M_CMD_DONE_EN | M_TX_FIFO_WATERMARK_EN;
	val |= M_RX_FIFO_WATERMARK_EN | M_RX_FIFO_LAST_EN;
	if (val != val_old) {
		writel(val, base + SE_GENI_M_IRQ_EN);
	}

	val_old = val = readl(base + SE_GENI_DMA_MODE_EN);
	val &= ~GENI_DMA_MODE_EN;
	if (val != val_old) {
		writel(val, base + SE_GENI_DMA_MODE_EN);
	}
}

u32 geni_se_get_tx_fifo_depth(phys_addr_t base, uint32_t geni_se_version)
{
	u32 tx_fifo_depth;
	u32 tx_fifo_depth_msk = TX_FIFO_DEPTH_MSK;

	if (geni_se_version >= QUP_SE_VERSION_3_10)
		tx_fifo_depth_msk = TX_FIFO_DEPTH_MSK_256_BYTES;

	tx_fifo_depth = ((readl(base + SE_HW_PARAM_0) &
				tx_fifo_depth_msk) >> TX_FIFO_DEPTH_SHFT);
	return tx_fifo_depth;

}

void geni_se_config_packing(phys_addr_t base, int bpw,
		bool msb_to_lsb, bool tx_cfg, bool rx_cfg)
{
	u32 cfg0, cfg1, cfg[NUM_PACKING_VECTORS] = {0};
	int len;
	int temp_bpw = bpw;
	int idx_start = msb_to_lsb ? bpw - 1 : 0;
	int idx = idx_start;
	int idx_delta = msb_to_lsb ? -BITS_PER_BYTE : BITS_PER_BYTE;
	int i, iter;
	unsigned int pack_words, ceil_bpw;

	if (bpw <= 8)
		pack_words = 4;
	else if (bpw <= 16)
		pack_words = 2;
	else
		pack_words = 1;

	ceil_bpw = (bpw & (BITS_PER_BYTE - 1)) ?
		((bpw & ~(BITS_PER_BYTE - 1)) + BITS_PER_BYTE) : bpw;

	iter = (ceil_bpw * pack_words) >> 3;
	if (iter <= 0 || iter > NUM_PACKING_VECTORS)
		return;

	for (i = 0; i < iter; i++) {
		len = min_t(int, temp_bpw, BITS_PER_BYTE) - 1;
		cfg[i] = idx << PACKING_START_SHIFT;
		cfg[i] |= msb_to_lsb << PACKING_DIR_SHIFT;
		cfg[i] |= len << PACKING_LEN_SHIFT;

		if (temp_bpw <= BITS_PER_BYTE) {
			idx = ((i + 1) * BITS_PER_BYTE) + idx_start;
			temp_bpw = bpw;
		} else {
			idx = idx + idx_delta;
			temp_bpw = temp_bpw - BITS_PER_BYTE;
		}
	}
	cfg[iter - 1] |= PACKING_STOP_BIT;
	cfg0 = cfg[0] | (cfg[1] << PACKING_VECTOR_SHIFT);
	cfg1 = cfg[2] | (cfg[3] << PACKING_VECTOR_SHIFT);

	if (tx_cfg) {
		writel(cfg0, base + SE_GENI_TX_PACKING_CFG0);
		writel(cfg1, base + SE_GENI_TX_PACKING_CFG1);
	}
	if (rx_cfg) {
		writel(cfg0, base + SE_GENI_RX_PACKING_CFG0);
		writel(cfg1, base + SE_GENI_RX_PACKING_CFG1);
	}

	/*
	 * Number of protocol words in each FIFO entry
	 * 0 - 4x8, four words in each entry, max word size of 8 bits
	 * 1 - 2x16, two words in each entry, max word size of 16 bits
	 * 2 - 1x32, one word in each entry, max word size of 32 bits
	 * 3 - undefined
	 */
	if (pack_words || bpw == 32) {
		writel(bpw / 16, base + SE_GENI_BYTE_GRAN);
	}
}

static int geni_se_qup_read(struct udevice *dev, int offset,
			    void *buf, int size)
{
	fdt_addr_t base = dev_read_addr(dev);

	if (size != sizeof(u32))
		return -EINVAL;

	*(u32 *)buf = readl(base + offset);

	return size;
}

static struct misc_ops geni_se_qup_ops = {
	.read = geni_se_qup_read,
};

static const struct udevice_id geni_se_qup_ids[] = {
	{ .compatible = "qcom,geni-se-qup" },
	{}
};

U_BOOT_DRIVER(geni_se_qup) = {
	.name = "geni_se_qup",
	.id = UCLASS_MISC,
	.of_match = geni_se_qup_ids,
	.ops = &geni_se_qup_ops,
	.flags  = DM_FLAG_PRE_RELOC,
};
