#include <common/debug.h>
#include <drivers/delay_timer.h>
#include <lib/mmio.h>
#include <platform_def.h>
#include "dram_fpga.h"

/* Memory test */
#define PATTERN1		0x5A5A5A5A
#define PATTERN2		0xA5A5A5A5

/* CVD may control following variables in the script */
int fpga_use_ddr2_256M = 0;
int fpga_use_ddr2_512M = 0;
int fpga_use_ddr4_8GB_x16 = 0;
int fpga_use_ddr4_8GB_x16_FDI = 0;
int fpga_use_ddr4_8GB_x32 = 0;
int fpga_dram_test = 0;
int fpga_dram_test_size = 0;

static int complex_mem_test(uintptr_t start, ssize_t len)
{
	uint8_t *mem8_base = (uint8_t *)start;
	uint16_t *mem16_base = (uint16_t *)start;
	uint32_t *mem32_base = (uint32_t *)start;
	uint8_t pattern8;
	uint16_t pattern16;
	uint32_t pattern32;
	uint32_t *mem_base = (uint32_t *)start;
	uint32_t value;
	ssize_t i, j, size;

	if (!len)
		return 0;

	size = len >> 2;

	/* Verify the tied bits (tied high) */
	for (i = 0; i < size; i++)
		mem32_base[i] = 0;

	for (i = 0; i < size; i++)
		if (mem32_base[i] != 0)
			return -1;
		else
			mem32_base[i] = 0xffffffff;

	/* Verify the tied bits (tied low) */
	for (i = 0; i < size; i++)
		if (mem32_base[i] != 0xffffffff)
			return -2;
		else
			mem32_base[i] = 0x00;

	/* Verify pattern 1 (0x00~0xff) */
	for (pattern8 = 0, i = 0; i < len; i++)
		mem8_base[i] = pattern8++;

	for (pattern8 = 0, i = 0; i < len; i++)
		if (mem8_base[i] != pattern8++)
			return -3;

	/* Verify pattern 2 (0x00~0xff) */
	for (pattern8 = 0, i = j = 0; i < len; i += 2, j++) {
		if (mem8_base[i] == pattern8)
			mem16_base[j] = pattern8;
		if (mem16_base[j] != pattern8)
			return -4;

		pattern8 += 2;
	}

	/* Verify pattern 3 (0x00~0xffff) */
	for (pattern16 = 0, i = 0; i < (len >> 1); i++)
		mem16_base[i] = pattern16++;

	for (pattern16 = 0, i = 0; i < (len >> 1); i++)
		if (mem16_base[i] != pattern16++)
			return -5;

	/* Verify pattern 4 (0x00~0xffffffff) */
	for (pattern32 = 0, i = 0; i < (len >> 2); i++)
		mem32_base[i] = pattern32++;

	for (pattern32 = 0, i = 0; i < (len >> 2); i++)
		if (mem32_base[i] != pattern32++)
			return -6;

	/* Pattern 5: Filling memory range with 0x44332211 */
	for (i = 0; i < size; i++)
		mem32_base[i] = 0x44332211;

	/* Read Check then Fill Memory with a5a5a5a5 Pattern */
	for (i = 0; i < size; i++)
		if (mem32_base[i] != 0x44332211)
			return -7;
		else
			mem32_base[i] = 0xa5a5a5a5;

	/* Read Check then Fill Memory with 00 Byte Pattern at offset 0h */
	for (i = 0; i < size; i++)
		if (mem32_base[i] != 0xa5a5a5a5)
			return -8;
		else
			mem8_base[i * 4] = 0x00;

	/* Read Check then Fill Memory with 00 Byte Pattern at offset 2h */
	for (i = 0; i < size; i++)
		if (mem32_base[i] != 0xa5a5a500)
			return -9;
		else
			mem8_base[i * 4 + 2] = 0x00;

	/* Read Check then Fill Memory with 00 Byte Pattern at offset 1h */
	for (i = 0; i < size; i++)
		if (mem32_base[i] != 0xa500a500)
			return -10;
		else
			mem8_base[i * 4 + 1] = 0x00;

	/* Read Check then Fill Memory with 00 Byte Pattern at offset 3h */
	for (i = 0; i < size; i++)
		if (mem32_base[i] != 0xa5000000)
			return -11;
		else
			mem8_base[i * 4 + 3] = 0x00;

	/* Read Check then Fill Memory with ffff Word Pattern at offset 1h */
	for (i = 0; i < size; i++)
		if (mem32_base[i] != 0x00000000)
			return -12;
		else
			mem16_base[i * 2 + 1] = 0xffff;

	/* Read Check then Fill Memory with ffff Word Pattern at offset 0h */
	for (i = 0; i < size; i++)
		if (mem32_base[i] != 0xffff0000)
			return -13;
		else
			mem16_base[i * 2] = 0xffff;

	/* Read Check */
	for (i = 0; i < size; i++)
		if (mem32_base[i] != 0xffffffff)
			return -14;

	/** Additional verification **/

	/* stage 1 => write 0 */
	for (i = 0; i < size; i++)
		mem_base[i] = PATTERN1;

	/* stage 2 => read 0, write 0xF */
	for (i = 0; i < size; i++) {
		value = mem_base[i];
		if (value != PATTERN1)
			return -15;

		mem_base[i] = PATTERN2;
	}

	/* stage 3 => read 0xF, write 0 */
	for (i = 0; i < size; i++) {
		value = mem_base[i];
		if (value != PATTERN2)
			return -16;

		mem_base[i] = PATTERN1;
	}

	/* stage 4 => read 0, write 0xF */
	for (i = 0; i < size; i++) {
		value = mem_base[i];
		if (value != PATTERN1)
			return -17;

		mem_base[i] = PATTERN2;
	}

	/* stage 5 => read 0xF, write 0 */
	for (i = 0; i < size; i++) {
		value = mem_base[i];
		if (value != PATTERN2)
			return -18;

		mem_base[i] = PATTERN1;
	}

	/* stage 6 => read 0 */
	for (i = 0; i < size; i++) {
		value = mem_base[i];
		if (value != PATTERN1)
			return -19;
	}

	/* 1/2/4-byte combination test */
	i = (ssize_t)mem_base;

	while (i < (ssize_t)mem_base + (size << 2)) {
		*((uint8_t *)i) = 0x78;
		i += 1;
		*((uint8_t *)i) = 0x56;
		i += 1;
		*((uint16_t *)i) = 0x1234;
		i += 2;
		*((uint32_t *)i) = 0x12345678;
		i += 4;
		*((uint16_t *)i) = 0x5678;
		i += 2;
		*((uint8_t *)i) = 0x34;
		i += 1;
		*((uint8_t *)i) = 0x12;
		i += 1;
		*((uint32_t *)i) = 0x12345678;
		i += 4;
		*((uint8_t *)i) = 0x78;
		i += 1;
		*((uint8_t *)i) = 0x56;
		i += 1;
		*((uint16_t *)i) = 0x1234;
		i += 2;
		*((uint32_t *)i) = 0x12345678;
		i += 4;
		*((uint16_t *)i) = 0x5678;
		i += 2;
		*((uint8_t *)i) = 0x34;
		i += 1;
		*((uint8_t *)i) = 0x12;
		i += 1;
		*((uint32_t *)i) = 0x12345678;
		i += 4;
	}

	for (i = 0; i < size; i++) {
		value = mem_base[i];
		if (value != 0x12345678)
			return -20;
	}

	return 0;
}

#pragma weak dram_lp2_init_MT7986_v2_init
void dram_lp2_init_MT7986_v2_init(void) {
	ERROR("No driver implementation for %s\n", __func__);
	panic();
}

#pragma weak dramc_init_Griffin_NPU_ddr4_3200_dphy_4BG_x16_init
void dramc_init_Griffin_NPU_ddr4_3200_dphy_4BG_x16_init(void) {
	ERROR("No driver implementation for %s\n", __func__);
	panic();
}

#pragma weak dramc_init_ddr4_3200_dphy_4BG_Griffin_NPU_x32_init
void dramc_init_ddr4_3200_dphy_4BG_Griffin_NPU_x32_init(void) {
	ERROR("No driver implementation for %s\n", __func__);
	panic();
}

#pragma weak dramc_init_Griffin_NPU_ddr4_3200_dphy_4BG_x16_FDI_init
void dramc_init_Griffin_NPU_ddr4_3200_dphy_4BG_x16_FDI_init(void) {
	ERROR("No driver implementation for %s\n", __func__);
	panic();
}

static void dramc_fpga_ddr2_dphy_256M_init(void) {
	dram_lp2_init_MT7986_v2_init();
	/* 256M */
	mmio_write_32(0x10219000, 0xa0a0a002);
	mmio_write_32(0x10236000, 0xa002);
}

static void dramc_fpga_ddr2_dphy_512M_init(void) {
	dram_lp2_init_MT7986_v2_init();
	/* 512M */
	mmio_write_32(0x10219000, 0xa0a05052);
	mmio_write_32(0x10236000, 0x5052);
}

void mtk_mem_init_fpga(void) {
	int ret;
	int kdram = 0;

	if (fpga_use_ddr2_256M) {
		NOTICE("EMI: start fpga ddr2 256M dram calibration\n");
		dramc_fpga_ddr2_dphy_256M_init();
		kdram = 1;
	}

	if (fpga_use_ddr2_512M) {
		NOTICE("EMI: start fpga ddr2 512M dram calibration\n");
		dramc_fpga_ddr2_dphy_512M_init();
		kdram = 1;
	}

	if (fpga_use_ddr4_8GB_x16) {
		NOTICE("EMI: start fpga ddr4 8GB_x16_V2 dram calibration\n");
		dramc_init_Griffin_NPU_ddr4_3200_dphy_4BG_x16_init();
		kdram = 1;
	}

	if (fpga_use_ddr4_8GB_x16_FDI) {
		NOTICE("EMI: start fpga ddr4 8GB_x16_FDI dram calibration\n");
		dramc_init_Griffin_NPU_ddr4_3200_dphy_4BG_x16_FDI_init();
		kdram = 1;
	}

	if (fpga_use_ddr4_8GB_x32) {
		NOTICE("EMI: start fpga ddr4 8GB_x32 dram calibration\n");
		dramc_init_ddr4_3200_dphy_4BG_Griffin_NPU_x32_init();
		kdram = 1;
	}

	if (kdram)
		INFO("EMI: fpga dram calibration finished\n");

	if (fpga_dram_test) {
		INFO("EMI: test dram between [0x40000000, 0x%x]\n",
		     (0x40000000 + fpga_dram_test_size));

		ret = complex_mem_test(0x40000000, fpga_dram_test_size);
		if (ret != 0) {
			ERROR("EMI: complex R/W mem test failed: %d\n", ret);
			panic();
		} else {
			INFO("EMI: complex R/W mem test passed\n");
		}
	}
}
